﻿using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.Shared;
using VA.TMP.Integration.VIMT.VirtualMeetingRoom.StateObject;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.VirtualMeetingRoom.PipelineSteps.Create
{
    /// <summary>
    /// Set the Meeting Room Name Step. 
    /// </summary>
    public class SetMeetingRoomNameStep : FilterBase<VirtualMeetingRoomCreateStateObject>
    {
        /// <summary>
        /// Execute the step.
        /// </summary>
        /// <param name="state">State object.</param>
        public override void Execute(VirtualMeetingRoomCreateStateObject state)
        {
            using (var context = new Xrm(state.OrganizationServiceProxy))
            {
                var serviceAppointment = context.ServiceAppointmentSet.First(x => x.Id == state.AppointmentId);

                if (serviceAppointment.ScheduledStart == null) throw new Exception("The Scheduled Start of the Service Appointment is null. Unable to continue processing.");

                // Get the date one hundred days prior to the Start of the Appointment.
                var dateHundredDaysAgo = serviceAppointment.ScheduledStart.Value.Subtract(new TimeSpan(100, 0, 0, 0));

                // In order to generate the Virtual Meeting Room name we have to have a random 7 digit number that does not already exist 100 days prior to the appointment.
                var allServiceAppointmentsScheduledForMonth = context.ServiceAppointmentSet
                    .Where(x => x.Id != state.AppointmentId && x.mcs_meetingroomname != null && x.ScheduledStart >= dateHundredDaysAgo && x.ScheduledStart <= serviceAppointment.ScheduledStart.Value)
                    .ToList()
                    .Where(x => x.mcs_meetingroomname.Length == (state.VirtualMeetingRoomPrefix.Length + state.VirtualMeetingRoomDigitLength))
                    .Select(x => x.mcs_meetingroomname.Substring(3, state.VirtualMeetingRoomDigitLength))
                    .ToDictionary(k => k, v => string.Empty);

                // If there are previous Meeting Rooms within the same month then call the helper method to find the next unique one; otherwise just generate it.
                state.MeetingRoomName = string.Format("{0}{1}", state.VirtualMeetingRoomPrefix, allServiceAppointmentsScheduledForMonth.Count > 0
                    ? GetNextVirtualMeetingRoomSuffix(allServiceAppointmentsScheduledForMonth, state.VirtualMeetingRoomDigitLength)
                    : RandomDigits.GetRandomDigitString(state.VirtualMeetingRoomDigitLength));
            }
        }

        /// <summary>
        /// Determine if a random string exists in the collection. If so then generate until there is a unique one.
        /// </summary>
        /// <param name="allServiceAppointmentsScheduledForMonth">Dictionary of Virtual Meeting Room Suffixes.</param>
        /// <param name="virtualMeetingRoomDigitLength">Length of Virtual Meeting Room prefix.</param>
        /// <returns>Random five digit numeric string that is not used for the current month.</returns>
        private static string GetNextVirtualMeetingRoomSuffix(IReadOnlyDictionary<string, string> allServiceAppointmentsScheduledForMonth, int virtualMeetingRoomDigitLength)
        {
            while (true)
            {
                var randomDigit = RandomDigits.GetRandomDigitString(virtualMeetingRoomDigitLength);

                if (allServiceAppointmentsScheduledForMonth.ContainsKey(randomDigit)) continue;

                return randomDigit;
            }
        }
    }
}
